/* -----------------------------------------------------------------
	EncDec EDNCODE Emulator
	Copyright (c) 2015-2016, by Dataman. All rights reserved.
	
	LATEST VERSION: ver. 6.4, updated 28th December 2015

   ----------------------------------------------------------------- */

/* OBJECT ----------------------------------------------------------
	Stores global variables and machine settings
   ----------------------------------------------------------------- */
function EDNCODE()
{
	// Debug the wiring.
	this.debug_wiring = 0; // 0 = off, 1 = on
	this.debug_string = "Kb  Pb  R   M   L   Rf  L   M   R   Pb\n"; // Init debug content string

	// Plaintext Alphabet
		this.plaintext = ".1234567890";

	// Rotor Wiring
		this.arrRotors = new Array();
		this.arrRotors[1] = ".0543298761"; // Rotor I
		this.arrRotors[2] = ".2175483609"; // Rotor II
		this.arrRotors[3] = ".5079183642"; // Rotor III
		this.arrRotors[4] = ".1284659370"; // Rotor IV
		this.arrRotors[5] = ".7432501896"; // Rotor V
		this.arrRotors[6] = ".3518209476"; // Rotor VI
		this.arrRotors[7] = ".0857394261"; // Rotor VII
		this.arrRotors[8] = ".2143658709"; // Rotor VIII
		this.arrRotors[9] = ".1234567890"; // Rotor IX
		this.arrRotors[10] = ".1234567890"; // Rotor X		
		this.arrRotors[11] = ".1234567890"; // Rotor XI		
		this.arrRotors[12] = ".1234567890"; // Rotor XII
		this.arrRotors["b"] = ".0284657391"; // M4 Greek Rotor "b" (beta)
		this.arrRotors["g"] = ".9876543210"; // M4 Greek Rotor "g" (gama)

	this.arrKnockpoints = new Array();
		this.arrKnockpoints[1] = new Array(6, 6, 6, 6, 6); //   Q - one knockpoint (R I)
		this.arrKnockpoints[2] = new Array(3, 3, 3, 3, 3); //   E - one knockpoint (R II)
		this.arrKnockpoints[3] = new Array(8, 8, 8, 8, 8); //   V - one knockpoint (R III)
		this.arrKnockpoints[4] = new Array(5, 5, 5, 5, 5); //   J - one knockpoint (R IV)
		this.arrKnockpoints[5] = new Array(10, 10, 10, 10, 10); //   Z - one knockpoint (R V)
		this.arrKnockpoints[6] = new Array(10, 5, 5, 5, 5); // Z/M - two knockpoints (R VI)
		this.arrKnockpoints[7] = new Array(10, 5, 5, 5, 5); // Z/M - two knockpoints (R VII)
		this.arrKnockpoints[8] = new Array(10, 7, 5, 5, 5); // Z/M - two knockpoints (R VIII)
		this.arrKnockpoints[9] = new Array(10, 8, 6, 4, 2); // Z/T/Q/N/G - five knockpoints (R IX)
		this.arrKnockpoints[10] = new Array(10, 10, 7, 7, 3); // Z/U/N/I/E - five knockpoints (R X)		
		this.arrKnockpoints[11] = new Array(10, 10, 5, 5, 5); // Z/S/M/H/C - five knockpoints (R XI)	
		this.arrKnockpoints[12] = new Array(0, 0, 0, 0, 0); // No Step - five knockpoints (R XII)
		
	// Reflectors "B" and "C" (inc M4 thin reflectors) Wiring
	this.arrReflector = new Array();
		this.arrReflector["a"] = ".0329785641"; // M3 A
		this.arrReflector["b"] = ".9543208716"; // M3 B
		this.arrReflector["c"] = ".2176043985"; // M3 C
		this.arrReflector["d"] = ".3918705426"; // M3 D		
		this.arrReflector["e"] = ".4801976253"; // M3 E
		this.arrReflector["f"] = ".7325481609"; // M3 F
		this.arrReflector["g"] = ".0987654321"; // M3 G Inv
		this.arrReflector["a_thin"] = ".5698120437"; // M4 thin A
		this.arrReflector["b_thin"] = ".6079813542"; // M4 thin B
		this.arrReflector["c_thin"] = ".7325481609"; // M4 thin C
		this.arrReflector["d_thin"] = ".8506249173"; // M4 thin D		

	this.usedletters = ""; // Used letters in the steckerbrett
	this.lastkeypressed = ""; // Keyboard key pressed
	this.counter = 0; // Number of letters entered for grouping

	this.inputmethod = "single";
}

/* FUNCTION --------------------------------------------------------
	Validate input (letter A-Z)
   ----------------------------------------------------------------- */
function validate(id)
{
	var string = document.getElementById(id).value;

	if (string.search(/['1'-'2'-'3'-'4'-'5'-'6'-'7'-'8'-'9'-'0']/gi)) {
		// If the input is not a letter clear field, focus on it and stop processing.
		document.getElementById(id).value = "";
		document.getElementById(id).focus();
		return false;
	}

	// Otherwise make it a capital letter.
	document.getElementById(id).value = string.toUpperCase();
	return true;
}

/* FUNCTION --------------------------------------------------------
	Swap the steckerbrett pairings.
	Eg, pair 'AD' - if 'A' goes in, 'D' comes out and vice versa
   ----------------------------------------------------------------- */
function swapPlugs(n)
{
	var plug_i = document.getElementById('plug'+n).value.toUpperCase();
	var plug_o = "";

	if (plug_i == "") {
		// If the input letter is blank (ie, self-steckered), output the letter unchanged.
		plug_o = n;
	} else {
		// Otherwise do the swapsies!
		plug_o = EDNCODE.plaintext.indexOf(plug_i);
	}

	return plug_o;
}

/* FUNCTION --------------------------------------------------------
	Validate and autofill the steckerbrett (plugboard)
   ----------------------------------------------------------------- */
function plugboard(n)
{
	var plugnum, re, otherletter;
	var arrusedletters = new Array();

	validate('plug'+n);

	var plug = document.getElementById('plug'+n).value.toUpperCase();

	if (plug != "") {
		// Check if letter has been used.
		if (EDNCODE.lastkeypressed >= 48 && EDNCODE.lastkeypressed <= 58 && EDNCODE.usedletters.indexOf(plug) > -1) {
			// If the input letter has already been used, ignore it and stop running the script.
			alert("You have already used the letter '"+plug+"' in a connection.\n"
					+ "Delete its current connection to form a new one.");
			document.getElementById('plug'+n).value = "";
			document.getElementById('plug'+n).focus();
			return false;
		} else {
			// Get the numerical value for the letter
			plugnum = EDNCODE.plaintext.indexOf(plug);

			// Fill out the paired letter field.
			//  Eg, if field 'A' is 'D', fill out field 'D' as 'A'
			document.getElementById('plug'+plugnum).value = EDNCODE.plaintext.charAt(n);

			EDNCODE.usedletters += plug + EDNCODE.plaintext.charAt(n) + " ";
		}
	} else {
		if (EDNCODE.lastkeypressed == 8 || EDNCODE.lastkeypressed == 46) {
			// Clear plugs on delete
			re = new RegExp("([A-Z]?)(" + EDNCODE.plaintext.charAt(n) + ")([1-0]?)\\s", "g");
			arrusedletters = re.exec(EDNCODE.usedletters);
			otherletter = arrusedletters[1] + arrusedletters[3];

			plugnum = EDNCODE.plaintext.indexOf(otherletter);
			document.getElementById('plug'+plugnum).value = "";
		
			re = new RegExp("([A-Z]?)" + EDNCODE.plaintext.charAt(n) + "([1-0]?)\\s", "g");
			EDNCODE.usedletters = EDNCODE.usedletters.replace(re, "");
		}
	}

	return true;
}

/* -----------------------------------------------------------------
	EDNCODE Process
	 1. Convert input letter to number - validate!
	 2. Rotate wheels
	 3. Pass through plugboard
	 4. Pass through right-hand wheel
	 5. Pass through middle wheel
	 6. Pass through left-hand wheel
	 7. Pass through reflector
	 8. Pass through left-hand wheel
	 9. Pass through middle wheel
	10. Pass through right-hand wheel
	11. Pass through plugboard
	12. Convert to output letter
   ----------------------------------------------------------------- */

/* FUNCTION --------------------------------------------------------
	Validate letter numbers (must be betwen 1 and 26)
	Remember, the real EDNCODE uses wheels/rotors
   ----------------------------------------------------------------- */
function validLetter(n)
{
	if (n <= 0) {
		// If negative number, add it to 26 to count back from "Z" (eg, 26 + -5 = 21)
		// Emulates wheel rotating backwards
		n = 10 + n;
	} else if (n >= 11) {
		// If number greater than 26, subtract 26 to count forward from "A" (eg, 30 - 26 = 4)
		// Emulates wheel rotating forwards
		n = n - 10;
	} else {
		// Or do nothing!
		n = n;
	}

	return n;
}

/* FUNCTION --------------------------------------------------------
	As text is entered, rotate the wheels (aka cogs) by one letter.
	This occurs BEFORE the entered text is mapped.
   ----------------------------------------------------------------- */
function rotateCogs(r,m,l)
{
	// r = right, m = middle, l = left

	var pr = EDNCODE.plaintext.indexOf(document.getElementById('rightw_set').value.toUpperCase());
	var pm = EDNCODE.plaintext.indexOf(document.getElementById('middlew_set').value.toUpperCase());
	var pl = EDNCODE.plaintext.indexOf(document.getElementById('leftw_set').value.toUpperCase());

	if (pr == parseFloat(EDNCODE.arrKnockpoints[r][0]) || pr == parseFloat(EDNCODE.arrKnockpoints[r][1]) || pr == parseFloat(EDNCODE.arrKnockpoints[r][2]) || pr == parseFloat(EDNCODE.arrKnockpoints[r][3]) || pr == parseFloat(EDNCODE.arrKnockpoints[r][4])) {
		// If the knockpoint on the right wheel is reached rotate middle wheel
		// But first check if it too is a knock point
		if (pm == parseFloat(EDNCODE.arrKnockpoints[m][0]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][1]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][2]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][3]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][4])) {
			// If the knockpoint on the middle wheel is reached rotate left wheel
			pl++;
		}
		pm++;
	} else {
		if (pm == parseFloat(EDNCODE.arrKnockpoints[m][0]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][1]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][2]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][3]) || pm == parseFloat(EDNCODE.arrKnockpoints[m][4])) {
			// If the knockpoint on the middle wheel is reached rotate left AND middle wheels
			// (the double stepping mechanism)
			pl++;
			pm++;
		}
	}

	// Rotate right wheel (this wheel is always rotated).
	pr++;

	// If rotating brings us beyond "Z" (26), then start at "A" (1) again.
	if (pr > 10) {pr = 1;}
	if (pm > 10) {pm = 1;}
	if (pl > 10) {pl = 1;}

	// Display new values in browser
	document.getElementById('rightw_set').value = EDNCODE.plaintext.charAt(pr);
	document.getElementById('middlew_set').value = EDNCODE.plaintext.charAt(pm);
	document.getElementById('leftw_set').value = EDNCODE.plaintext.charAt(pl);

	// Make values available to the rest of the script as an array.
	return new Array(pr, pm, pl);
}

/* FUNCTION --------------------------------------------------------
	Map one letter to another through current wheel
   ----------------------------------------------------------------- */
function mapLetter(number, ringstellung, wheelposition, wheel, pass)
{
	// Variables: number = input letter; ringstellung = wheel ring setting (static);
	// wheelposition = wheel position (rotates); wheel = current wheel;
	// pass = are we going R->L (1) or L->R (2)

	// Change number according to ringstellung (ring setting)
	// Wheel turns anti-clockwise (looking from right)

	number = number - ringstellung;

		// Check number is between 1 and 26
		number = validLetter(number);

	// Change number according to wheel position
	// Wheel turns clockwise (looking from right)

	number = number + wheelposition;

		// Check number is between 1 and 26
		number = validLetter(number);

	// Do internal connection 'x' to 'y' according to direction  
	if (pass == 2) {
		var let = EDNCODE.plaintext.charAt(number);
		number = EDNCODE.arrRotors[wheel].indexOf(let);
	} else {
		var let = EDNCODE.arrRotors[wheel].charAt(number);
		number = EDNCODE.plaintext.indexOf(let);
	}

	// 
	// NOW WORK IT BACKWARDS : subtract where we added and vice versa
	// 

	// Change according to wheel position (anti-clockwise)

	number = number - wheelposition;

		// Check number is between 1 and 26
		number = validLetter(number);

	// Change according to ringstellung (clockwise)

	number = number + ringstellung;

		// Check number is between 1 and 26
		number = validLetter(number);

	return number;
}

/* FUNCTION --------------------------------------------------------
	Light up the output letter!
	Unnecessary (but cute) imitation of the real EDNCODE machines
   ----------------------------------------------------------------- */
function lightUp(n)
{
	for (var i = 1; i <= 10; i++) {
	  // Wipe all previous lights - make font colour grey
	  document.getElementById('l_'+EDNCODE.plaintext.charAt(i)).setAttribute('class', 'led_off');
	}

	// Light up the outout letter - make font colour yellow
	document.getElementById('l_'+n).setAttribute('class', 'led_on'); 
}

/* FUNCTION --------------------------------------------------------
	Debug wiring - capture current state of text at change points
	Only displayed to screen if debug_wiring is on.
   ----------------------------------------------------------------- */
function run_debug(m, n)
{
	if (m == 1) {
		EDNCODE.debug_string += EDNCODE.plaintext.charAt(n);
		if (EDNCODE.debug_wiring == 1) {
			document.getElementById('showDebug').value += EDNCODE.debug_string + "\n";
		}
		EDNCODE.debug_string = "";
	} else {
		EDNCODE.debug_string += EDNCODE.plaintext.charAt(n) + " > ";
	}
}

/* FUNCTION --------------------------------------------------------
	Run the whole EDNCODE machine process.
	The functions above are called from this routine.
   ----------------------------------------------------------------- */
function doCipher()
{
	// Type of EDNCODE to Use: M3/M4
	var enigma_type = document.getElementById('enigma_type').value;

	// Get current status of Wheel Order
	var wheel_r = parseFloat(document.getElementById('rightw_no').value);
	var wheel_m = parseFloat(document.getElementById('middlew_no').value);
	var wheel_l = parseFloat(document.getElementById('leftw_no').value);
	var wheel_g = document.getElementById('greek_no').value; // M4 Greek Wheek

	// Get current status of Wheel Ring Setting
	var ring_r = EDNCODE.plaintext.indexOf(document.getElementById('rightw_ring').value.toUpperCase());
	var ring_m = EDNCODE.plaintext.indexOf(document.getElementById('middlew_ring').value.toUpperCase());
	var ring_l = EDNCODE.plaintext.indexOf(document.getElementById('leftw_ring').value.toUpperCase());
	var ring_g = EDNCODE.plaintext.indexOf(document.getElementById('greek_ring').value.toUpperCase());

	// If using M4 EDNCODE
	if (enigma_type == "m4") {
		// Which reflector are we using? B thin (default) or C tbin
		var useReflector = document.getElementById('use_reflector_thin').value;
	} else {
		// Which reflector are we using? B (default) or C
		var useReflector = document.getElementById('use_reflector').value;
	}

	// Grouping of letters in blocks
	var grouping = document.getElementById('letter_grouping').value;

	// Are the selected rotors all different?
	if (wheel_r == wheel_m || wheel_r == wheel_l || wheel_m == wheel_l ) {
		alert("Wheel Numbers must be unique. Eg, I II III not II II II");
		document.getElementById('rightw_no').focus();
		return false;
	}

	// Get input letter
	var letterinput = document.getElementById('textin').value.toUpperCase();

	if (letterinput.search(/['1'-'2'-'3'-'4'-'5'-'6'-'7'-'8'-'9'-'0']/gi)) {
		// If input is not a letter [A-Z], then return false and do nothing
		// except clear and focus the letter input field
		document.getElementById('textin').value = "";
		document.getElementById('textin').focus();
		return false;
	}

	// Rotate Wheels
	var wheel_position = rotateCogs(wheel_r,wheel_m,wheel_l);

	// Wheel Starting Position
	var start_r = wheel_position[0];
	var start_m = wheel_position[1];
	var start_l = wheel_position[2];
	var start_g = EDNCODE.plaintext.indexOf(document.getElementById('greek_set').value.toUpperCase()); // M4 does not rotate

	// Input
	var input = EDNCODE.plaintext.indexOf(letterinput);

	run_debug(0, input);

	// First Pass - Plugboard
	var number = swapPlugs(input);

	run_debug(0, number);

	// Passes through ETW which acts as a static converter from plugboard wires to wheels
	// So:  Plugboard --> ETW --> Right Wheel
	// A -->  A  --> A

	// First Pass - R Wheel
	number = mapLetter(number,ring_r,start_r,wheel_r, 1);

	run_debug(0, number);

	// First Pass - M Wheel
	number = mapLetter(number,ring_m,start_m,wheel_m, 1);

	run_debug(0, number);

	// First Pass - L Wheel
	number = mapLetter(number,ring_l,start_l,wheel_l, 1);

	run_debug(0, number);

	// If using M4 EDNCODE
	if (enigma_type == "m4") {
		// First Pass - Greek Wheel
		number = mapLetter(number,ring_g,start_g,wheel_g, 1);
	}

	// Reflector
	var let = EDNCODE.arrReflector[useReflector].charAt(number);
	number = EDNCODE.plaintext.indexOf(let);

	run_debug(0, number);

	// If using M4 EDNCODE
	if (enigma_type == "m4") {
		// Second Pass - Greek Wheel
		number = mapLetter(number,ring_g,start_g,wheel_g, 2);
	}

	// Second Pass - L Wheel
	number = mapLetter(number,ring_l,start_l,wheel_l, 2);

	run_debug(0, number);

	// Second Pass - M Wheel
	number = mapLetter(number,ring_m,start_m,wheel_m, 2);

	run_debug(0, number);

	// Second Pass - R Wheel
	number = mapLetter(number,ring_r,start_r,wheel_r, 2);

	run_debug(0, number);

	// Passes through ETW again

	// Second Pass - Plugboard
	number = swapPlugs(number);

	run_debug(1, number);

	// Convert value to corresponding letter
	var output = EDNCODE.plaintext.charAt(number);

	// Clean number
	number = "";

	// Build Message Strings for Input and Output

	// Get current string values
	var msg_in = document.getElementById('msg_in').value;
	var msg_out = document.getElementById('msg_out').value;

	if (EDNCODE.counter == grouping) {
		// Space out message in/out as letter blocks of X length (grouping)
		msg_in = msg_in + " ";
		msg_out = msg_out + " ";
		EDNCODE.counter = 0;
	}

	// Increment counter
	EDNCODE.counter ++;

	// Spit out new string values
	document.getElementById('msg_in').value = msg_in + letterinput;
	document.getElementById('msg_out').value = msg_out + output;

	// Show output letter in lightbox.
	lightUp(output);

	// Clear and focus letter input field
	document.getElementById('textin').value = "";
	document.getElementById('textin').focus();

	return true;
}


/* -----------------------------------------------------------------
	Code for saving & loading machine settings.
	Not part of main EDNCODE emulation
   ----------------------------------------------------------------- */


/* FUNCTION --------------------------------------------------------
	Load a saved user preset
   ----------------------------------------------------------------- */
function loadPreset(strPre)
{
	if ( strPre.length > 0 ) {
		// Str format: m3;b;b123;AAAA;AAAA
		
		var strBeg = 0;
		var strEnd = strPre.length;
		var strVar = strPre.substring(strBeg,strEnd);

		var arrVars = strVar.split(";");

		var mac = arrVars[0]
		var ukw = arrVars[1];
		var rotors = arrVars[2];
		var rings = arrVars[3];
		var ground = arrVars[4];
	
		document.getElementById('enigma_type').value = mac;

		if (mac == "m4") {
			document.getElementById('use_reflector_thin').value = ukw;
		} else {
			document.getElementById('use_reflector').value = ukw;
		}

		document.getElementById('greek_no').value = rotors.charAt(0);
		document.getElementById('leftw_no').value = rotors.charAt(1);
		document.getElementById('middlew_no').value = rotors.charAt(2);
		document.getElementById('rightw_no').value = rotors.charAt(3);

		document.getElementById('greek_ring').value = rings.charAt(0);
		document.getElementById('leftw_ring').value = rings.charAt(1);
		document.getElementById('middlew_ring').value = rings.charAt(2);
		document.getElementById('rightw_ring').value = rings.charAt(3);

		document.getElementById('greek_set').value = ground.charAt(0);
		document.getElementById('leftw_set').value = ground.charAt(1);
		document.getElementById('middlew_set').value = ground.charAt(2);
		document.getElementById('rightw_set').value = ground.charAt(3);

		var n = 1;
		while (n <= 10) {
			document.getElementById('plug'+n).value = ""; 
			n++;
		}

		if (arrVars.length > 5) {
			var arrStecker = arrVars[5].split("-");
			var i = 0;
			while (i < arrStecker.length) {
				var a = arrStecker[i].charAt(0).toUpperCase();
				var b = arrStecker[i].charAt(1).toUpperCase();
				var an = EDNCODE.plaintext.indexOf(a);
				var bn = EDNCODE.plaintext.indexOf(b);
				document.getElementById('plug'+an).value = b;
				document.getElementById('plug'+bn).value = a;
				i++;
			}
		}
	}
}

/* FUNCTION --------------------------------------------------------
	Save a user preset as a bookmark
   ----------------------------------------------------------------- */
function savePreset()
{
	var mac = document.getElementById('enigma_type').value;

	// If using M4 EDNCODE
	if (mac == "m4") {
		var ukw = document.getElementById('use_reflector_thin').value + ";";
	} else {
		var ukw = document.getElementById('use_reflector').value + ";";
	}

	mac = 	mac + ";"

	var rot = document.getElementById('greek_no').value
				 + document.getElementById('leftw_no').value 
				 + document.getElementById('middlew_no').value
				 + document.getElementById('rightw_no').value + ";";

	var rin = document.getElementById('greek_ring').value
				+ document.getElementById('leftw_ring').value
				+ document.getElementById('middlew_ring').value
				+ document.getElementById('rightw_ring').value + ";";

	var gro = document.getElementById('greek_set').value
				 + document.getElementById('leftw_set').value
				 + document.getElementById('middlew_set').value
				 + document.getElementById('rightw_set').value;

	var stk = "";
	var i = 1;

	while (i <= 10) {
		if (document.getElementById('plug'+i).value != "") {
			var string_rev = document.getElementById('plug'+i).value.toUpperCase() + EDNCODE.plaintext.charAt(i);

			if (stk.indexOf(string_rev) < 0) {
				if (stk != "") {
					stk = stk + "-";
				} else {
					stk = ";" + stk;
				}
				stk = stk + EDNCODE.plaintext.charAt(i) + document.getElementById('plug'+i).value.toUpperCase();
			}
		}
		i++;
	}

	var months = new Array("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec");

	var now = new Date();
	var title = now.getDate() + " " + months[now.getMonth()] + ", " + now.getFullYear() + " "
	+ now.getHours() + ":"
	+ ((now.getMinutes() < 10) ? "0" : "") + now.getMinutes() + ":"
	+ ((now.getSeconds() < 10) ? "0" : "") + now.getSeconds(); 
	
	var url = document.URL.replace(/\?.*/gi, "") + "?" + mac + ukw + rot + rin + gro + stk;;

	var savearea = document.getElementById('savearea');

	// Quick'n'dirty use of inner HTML, so sue me!
	savearea.innerHTML = "<p><a href=\"" + url  + "\">EDNCODE Settings: " + title + "</a><br />\n"
		+ "Right-click on the above link and select \"Add to Bookmarks\" (or similar), or "
		+ "drag it to your bookmarks folder.</p>";

	savearea.setAttribute("class", "showsave");

}

/* FUNCTION --------------------------------------------------------
	Tidy up after the reset button is pressed
   ----------------------------------------------------------------- */
function tidyUp()
{
	var i = 1;
	while (i <= 10) {
	  // Wipe all previous lights - make font colour grey
	  document.getElementById('l_'+EDNCODE.plaintext.charAt(i)).setAttribute('class', 'led_off'); 
	  i++;
	}

	document.getElementById('save').removeAttribute('class');

	var muckyurl = document.URL;
	var cleanurl = muckyurl.replace(/\?.*/gi, "");
	document.location = cleanurl;

	EDNCODE.usedletters = "";
	EDNCODE.lastkeypressed = "";
	EDNCODE.counter = 0;
}

/* FUNCTION --------------------------------------------------------
	Initialise Engima and set-up onload behaviours
   ----------------------------------------------------------------- */
function initEDNCODE()
{
	// Do nothing if the DOM not supported
	if (!document.getElementById || !document.getElementsByTagName) { return false; }

	// Hide the non-DOM message
	document.getElementById('jsmsg').style.display = "none";

	EDNCODE = new EDNCODE(); // Grabs global variables as EDNCODE object

	// Local vars
	var i, re, wid, tid, currstatus;

	// Add a configure action to EDNCODE type selection
	var etype = document.getElementById('enigma_type');
	etype.onchange = function() {
		if (this.value == "m4") {
			document.getElementById('greek_no').style.display = "";
			document.getElementById('greek_ring').style.display = "";
			document.getElementById('greek_set').style.display = "";
			document.getElementById('use_reflector_thin').style.display = "";
			document.getElementById('urt').style.display = "";
			document.getElementById('use_reflector').style.display = "none";
			document.getElementById('ur').style.display = "none";

		}
		else
		{
			document.getElementById('greek_no').style.display = "none";
			document.getElementById('greek_ring').style.display = "none";
			document.getElementById('greek_set').style.display = "none";
			document.getElementById('use_reflector_thin').style.display = "none";
			document.getElementById('urt').style.display = "none";
			document.getElementById('use_reflector').style.display = "";
			document.getElementById('ur').style.display = "";
		}
	}

	// Wheel Settings
	var init_wheels = document.getElementById('wheels');
	var iw = init_wheels.getElementsByTagName('input');

	for (i = 0; i < iw.length; i++) {
		// Add validation function to wheels
		if (iw[i].getAttribute('type') == "text") {
			iw[i].onkeyup = function() {
				wid = this.getAttribute('id');
				validate(wid);
			}
		}
	}

	// Plugboard setting
	var init_plugboard = document.getElementById('plugboard');
	var ip = init_plugboard.getElementsByTagName('input');

	for (i = 0; i< ip.length; i++) {

		ip[i].onkeydown = function() {
			currstatus = this.value;
		}

		// Add the swap plugs function
		ip[i].onkeyup = function() {
			re = /plug([0-9]+)/i
			tid = this.getAttribute('id').replace(re, "$1");

			if (currstatus == "" || EDNCODE.lastkeypressed == 8 || EDNCODE.lastkeypressed == 46) {
				plugboard(tid);
				currstatus = this.value;
			} else {
				if (EDNCODE.lastkeypressed >= 48 && EDNCODE.lastkeypressed <= 58) {
					this.value = currstatus;
				}
			}
		}
	}

	// Add a the getPreset function to button
	var init_save = document.getElementById('save');
	init_save.onclick = function() {
		if (this.getAttribute('class') == "nosave") return false;
		savePreset();
		return false;
	}

	// Add a the tidyUp function to button
	var init_reset = document.getElementById('clearup');
	init_reset.onclick = function() {
		tidyUp();
		document.getElementById('EDNCODE').reset();
		return false;
	}

	// Set input/output fields to readonly
	document.getElementById('msg_in').setAttribute('readonly', 'readonly');
	document.getElementById('msg_out').setAttribute('readonly', 'readonly');

	// If debug_wiring is on, show form field
	if (EDNCODE.debug_wiring == 1)
	{
		var debug_f = document.getElementById('EDNCODE');
		var debug_p = document.createElement('p')
		var debug_l = document.createTextNode('Debug Window:');
		var debug_b = document.createElement('br');
		var debug_t = document.createElement('textarea');
		debug_t.id = "showDebug";
		debug_t.cols = 40;
		debug_t.rows = 20;
		debug_f.appendChild(debug_p);
		debug_p.appendChild(debug_l);
		debug_p.appendChild(debug_b);
		debug_p.appendChild(debug_t);
	}

	// Are we loading a saved preset?
	var url = document.URL;
	if (url.indexOf('?') > -1) {
		var str_settings = url.substring(url.indexOf("?")+1, url.length);
		loadPreset(str_settings);
	}

	if (document.getElementById('enigma_type').value == "m3") {
		document.getElementById('greek_no').style.display = "none";
		document.getElementById('greek_ring').style.display = "none";
		document.getElementById('greek_set').style.display = "none";
		document.getElementById('urt').style.display = "none";
		document.getElementById('use_reflector_thin').style.display = "none";
	} else {
		document.getElementById('ur').style.display = "none";
		document.getElementById('use_reflector').style.display = "none";
	}

	// Add onclick functions to "single" or "block" input radio buttons

	if (EDNCODE.inputmethod == "single") {
		document.getElementById('blockinput').style.display = "none";
	}
	if (EDNCODE.inputmethod == "block") {
		document.getElementById('singleinput').style.display = "none";
	}
	
	var method_single = document.getElementById('single');
	var method_block = document.getElementById('block');

	method_single.onclick = function() {
		document.getElementById('blockinput').style.display = "none";
		document.getElementById('singleinput').style.display = "";
		EDNCODE.inputmethod = "single";
	}
	method_block.onclick = function() {
		document.getElementById('blockinput').style.display = "";
		document.getElementById('singleinput').style.display = "none";
		EDNCODE.inputmethod = "block";
	}

	// Enter letter - add the cipher function
	var init_txtinput = document.getElementById('textin');
	init_txtinput.onkeyup = function() {
		if (EDNCODE.inputmethod == "single")
		{
			doCipher();
			if (EDNCODE.lastkeypressed >= 65 && EDNCODE.lastkeypressed <= 90) {
				document.getElementById('save').setAttribute('class', 'nosave');
			}
		}
	}

	var init_startbtn = document.getElementById('startbtn');
	var temp = document.getElementById('textin');
	init_startbtn.onclick = function() {
		if (EDNCODE.inputmethod == "block")
		{
			var init_txtinput = document.getElementById('txt_in').value;
			for (i = 0; i < init_txtinput.length; i++)
			{
				temp.value = init_txtinput.charAt(i);
				doCipher();

				// Break after every 500 chars to stop overloading browser
				// Also stops somebody breaking it by pasting in a novel!
				if (i > 0 && i % 800 == 0) {
					var cont = confirm("The EDNCODE Machine is having a breather...\n\n"
						+ "Continue with the rest of the cipher?");

					if (cont == false) {
						break;
					}
				}
			}
			document.getElementById('txt_in').value = ""; // Clear field or next input block
		}
	}
}

/* FUNCTION --------------------------------------------------------
	Store the keypress
   ----------------------------------------------------------------- */
function getKeyIn(key)
{
	if (!key) {
		key = event;
		key.which = key.keyCode;
	}
	EDNCODE.lastkeypressed = key.which;
}


/* FUNCTION --------------------------------------------------------
	Add and Remove Events by John Resig
	http://ejohn.org/projects/flexible-javascript-events/
   ----------------------------------------------------------------- */
function addEvent( obj, type, fn )
{
	if ( obj.attachEvent ) {
		obj["e"+type+fn] = fn;
		obj[type+fn] = function() { obj["e"+type+fn]( window.event ); }
		obj.attachEvent( "on"+type, obj[type+fn] );
	} else {
		obj.addEventListener( type, fn, false );
	}
}
function removeEvent( obj, type, fn )
{
	if ( obj.detachEvent ) {
		obj.detachEvent( "on"+type, obj[type+fn] );
		obj[type+fn] = null;
	} else {
		obj.removeEventListener( type, fn, false );
	}
}

addEvent(window, "load", initEDNCODE);
addEvent(document, "keydown", getKeyIn);
